﻿using System;
using System.Collections.Generic;
using System.Configuration;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Xml.Linq;

namespace gov.va.med.vbecs.Common.Configuration
{
    /// <summary>
    /// AppSettingsReade that reads configuration values from XML file
    /// </summary>
    public class AppSettingsReaderXml : IAppSettingsReader
    {
        /// <summary>
        /// Application name. Assigned by Spring.NET
        /// </summary>
        public string AppName { get; internal set; }

        /// <summary>
        /// Indicates if it is a service. Assigned by Spring.NET
        /// </summary>
        public bool IsService { get; internal set; }

        /// <summary>
        /// DataSourceString
        /// </summary>
        public string DataSourceString { get; internal set; }

        /// <summary>
        /// SettingsMap
        /// </summary>
        protected IDictionary<string, string> SettingsMap;
        static readonly Type StringType = typeof(string);
        private const string NULL_STRING = "None";

        /// <summary>
        /// AppSettingsReaderXml
        /// </summary>
        internal AppSettingsReaderXml()
        {}

        /// <summary>
        /// GetValue
        /// </summary>
        /// <param name="key"></param>
        /// <param name="type"></param>
        /// <returns></returns>
        public object GetValue(string key, Type type)
        {
            if(SettingsMap == null)
                SettingsMap = read_application_settings(AppName);

            if (key == null) throw new ArgumentNullException("key");
            if(!SettingsMap.ContainsKey(key))
                throw new ArgumentException(string.Format("Provided Key is not found in the configuration: {0}, for application: {1}", key, AppName));

            string val = SettingsMap[key];
            if (val == null) throw new InvalidOperationException("AppSettingsReader: no key");


            // Gets the value for specified key from ConfigurationManager.AppSettings, and returns
            // an object of the specified type containing the value from the config file.  If the key 
            // isn't in the config file, or if it is not a valid value for the given type, it will 
            // throw an exception with a descriptive message so the user can make the appropriate change 
            if (type == StringType)
            {
                // It's a string, so we can ALMOST just return the value.  The only
                // tricky point is that if it's the string "(None)", then we want to
                // return null.  And of course we need a way to represent the string 
                // (None), so we use ((None)), and so on... so it's a little complicated.
                int noneNesting = get_none_nesting(val);
                if (noneNesting == 0)
                {
                    // val is not of the form ((..((None))..))
                    return val;
                }
                if (noneNesting == 1)
                {
                    // val is (None)
                    return null;
                }
                // val is of the form ((..((None))..)) 
                return val.Substring(1, val.Length - 2);
            }
            try
            {
                return Convert.ChangeType(val, type, CultureInfo.InvariantCulture);
            }
            catch (Exception)
            {
                throw new InvalidOperationException(string.Format(
                    "Can't parse [{0}] string for Key=[{1}] to Type=[{2}]", val, key, type)); 
            }
        }

        private int get_none_nesting(string val)
        {
            int count = 0;
            int len = val.Length;
            if (len > 1)
            {
                while (val[count] == '(' && val[len - count - 1] == ')')
                {
                    count++;
                }
                if (count > 0 && string.Compare(NULL_STRING, 0, val, count, len - 2 * count, StringComparison.Ordinal) != 0)
                {
                    // the stuff between the parents is not "None"
                    count = 0;
                }
            }
            return count;
        }

        private IDictionary<string, string> read_application_settings(string appName)
        {
            XDocument xdoc = XDocument.Load(Path.Combine(Path.GetDirectoryName(typeof(AppSettingsReaderXml).Assembly.Location) ?? ".", DataSourceString));

            var results =
                xdoc.Descendants("application").Attributes("name").Where(e => e.Value == appName).Select(e => e.Parent).
                    Descendants("add").Select(e => new { name = e.Attribute("key"), value = e.Attribute("value") }).ToList();

            // Check if there are any null keys
            if (results.Any(x => x.name == null || x.value == null))
                throw new ConfigurationErrorsException(string.Format("Some keys or values in the configuration file [{0}] are NULL for application [{1}]", DataSourceString, appName));
            return results.Select(x => new { name = x.name.Value, value = x.value.Value }).ToDictionary(kvp => kvp.name, kvp => kvp.value);
        }

    }
}
